<?php

namespace App\Http\Controllers\Api;

use App\Models\Advertisement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class AdvertisementController
{
    /**
     * Display a listing of the resource.
     */
    public function getAll()
    {
        $advertisement = Advertisement::all();

        $advertisement->transform(function($item){
            $item->multimedia = asset('storage/' . $item->multimedia);
            return $item;
        });

        // Return as JSON response with UTF-8 encoding
        return response()->json($advertisement, 200, ['Content-Type' => 'application/json; charset=UTF-8'], JSON_UNESCAPED_UNICODE);
        
    }

    public function getRandom()
    {
        
        // Obtener todos los IDs disponibles en un array
        $ids = Advertisement::pluck('id')->toArray();
        
    
        // Verificar si hay anuncios disponibles
        if (count($ids) === 0) {

            return response()->json(['message' => 'No hay publicidades disponibles'], 404);
        }
    
        // Seleccionar un ID aleatorio
        $randomId = $ids[array_rand($ids)]; 
    
        // Obtener la publicidad correspondiente al ID aleatorio
        $advertisement = Advertisement::where('id', $randomId)->first(); // Usar first() en vez de find()
    
        // Verificar si se encontró el anuncio
        if (!$advertisement) {
            return response()->json(['message' => 'No se encontró la publicidad'], 404);
        }
    
        // Convertir la URL de multimedia si existe
        if (!empty($advertisement->multimedia)) {
            $advertisement->multimedia = asset('storage/' . $advertisement->multimedia);
        }
    
        return response()->json($advertisement, 200, ['Content-Type' => 'application/json; charset=UTF-8'], JSON_UNESCAPED_UNICODE);
    }
    

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //check values
        $validatedData = Validator::make($request->all(),[
            'titulo'=>'required|string|max:255',
            'url'=>'required|string',
            'fecha_activacion'=>'required|date',
            'fecha_fin'=>'required|date',
            'fecha_publicacion'=>'required|date',
            'id_administrador'=>'required|string',
        ]);

        try{
                $file = $request->file('multimedia');
                $path = $file->store('uploads/advertisement', 'public');

                if ($validatedData->fails()) {
                    return response()->json([
                    'message' => 'Validation failed',
                    'errors' => $validatedData->errors() ],
                    422);
                }

                $advertisementCreated = DB::table('advertisements')->insert([
                    'titulo'=> $request->titulo,
                    'url'=> $request->url, 
                    'multimedia' => $path,
                    'fecha_activacion' => $request->fecha_activacion,
                    'fecha_fin' => $request->fecha_fin,
                    'fecha_publicacion' => $request->fecha_publicacion,
                    'id_administrador' => $request->id_administrador
                ]);

                return response()->json(
                    ['message' => 'Se creó la publicidad correctamente'],
                    200,
                    ['Content-Type' => 'application/json; charset=UTF-8'],
                    JSON_UNESCAPED_UNICODE);

            }catch(Exception $error){
                    return response()->json(
                      "Error al realizar la consulta crea publicidad: " . $error->getMessage(),
                      500,
                      ['Content-Type' => 'application/json; charset=UTF-8'],
                      JSON_UNESCAPED_UNICODE
                    );
                  }
        
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //Get advertisementg where $id matches
        $advertisement = Advertisement::where('id', $id)->get();

        return response()->json($advertisement, 200,['Content-Type' => 'application/json; charset=UTF-8'], JSON_UNESCAPED_UNICODE);
    }

    /**
     * Update the specified resource in storage.
     */

     public function update(Request $request, string $id) // Se especifica que $id es un string
     {   
     
         // Validar que el ID es una cadena no vacía
         if (!is_string($id) || empty($id)) {
             return response()->json([
                 'message' => 'El ID proporcionado no es válido'
             ], 400);
         }
     
         // Validación de datos del request
         $validatedData = Validator::make($request->all(), [
             'titulo' => 'required|string|max:255',
             'url' => 'required|string',
             'fecha_activacion' => 'required|date',
             'fecha_fin' => 'required|date',
             'fecha_publicacion' => 'required|date',
             'id_administrador' => 'required|string',
         ]);
     
         if ($validatedData->fails()) {
             return response()->json([
                 'message' => 'Validation failed',
                 'errors' => $validatedData->errors()
             ], 422);
         }
     
         try {
             // Buscar la publicidad en la base de datos
             $advertisement = DB::table('advertisements')->where('id', $id)->first();
     
             if (!$advertisement) {
                 return response()->json([
                     'message' => "No se encontró la publicidad con el ID: $id"
                 ], 404);
             }
     
             // Mantener la ruta del archivo multimedia si no hay nuevo archivo
             $path = $advertisement->multimedia;
             $file = $request->file('multimedia');
     
             if ($file) {
                 // Subir nuevo archivo
                 $path = $file->store('uploads/advertisement', 'public');
     
                 // Eliminar archivo anterior si existe
                 if ($advertisement->multimedia && Storage::disk('public')->exists($advertisement->multimedia)) {
                     Storage::disk('public')->delete($advertisement->multimedia);
                 }
             }
     
             // Actualizar registro en la base de datos
             DB::table('advertisements')
                 ->where('id', $id)
                 ->update([
                     'titulo' => $request->titulo,
                     'url' => $request->url,
                     'multimedia' => $path,
                     'fecha_activacion' => $request->fecha_activacion,
                     'fecha_fin' => $request->fecha_fin,
                     'fecha_publicacion' => $request->fecha_publicacion,
                     'id_administrador' => $request->id_administrador
                 ]);
     
             return response()->json([
                 'message' => 'Se editó la publicidad correctamente'
             ], 200);
     
         } catch (Exception $error) {
 
             return response()->json([
                 'message' => 'Error al editar publicidad',
                 'error' => $error->getMessage()
             ], 500);
         }
     }
     
     


    /**
     * Remove the specified resource from storage.
     */
    public function deleteAdvertisement(string $id)
    {
      $advertisement_id = $id;

      if (!$advertisement_id){
          return response()->json(['message' => 'No se encontro el registro'],404);
      }

      //delete requirement
      $deleted = Advertisement::where('id', $advertisement_id)->delete();

      //Delete
      if ($deleted){
          return response()->json(['message' => 'Se eliminó la publicidad'],200);
      }else{
          return response()->json(['message' => 'Fallo en la eliminación de la publicidad'],400);
      }
    }

}
